turtles-own [start-patch] ;; original position of ball on kick line
patches-own [score        ;; score for this position along the kick line
             left-angle   ;; angle towards left goal-post
             right-angle  ;; angle towards right goal-post
             goal-angle   ;; size of arc between left-angle and right-angle
             slope]       ;; slope of line from this patch towards a goal-post
globals [current-max      ;; the best patch-score so far
         col              ;; current color for our level-curves
         ang              ;; viewing angle of the current level curve
         best-dist        ;; distance from try-line of best kick
         analytic         ;; what the best distance should be, analytically
         try-line         ;; agentset containing only those patches on the try line
         histogram-area   ;; agentset containing only patches inside the histogram
         kicks            ;; total number of balls kicked
         goals]           ;; total number of goals scored

to setup
  ca
  setup-field
  setup-balls
  set current-max 0
  set best-dist -1
  set kicks 0
  ask try-line [ set score 0 ]
  find-analytic-solution
  if show-level-curves? [ draw-level-curves ]
end

to setup-field
  ;; Draw lines for border, kick line, and goal line
  ask patches [
    if ((abs pxcor) = screen-edge-x) or ((abs pycor) = screen-edge-y)
     [set pcolor red ]
    if (pycor = screen-edge-y) and
       (pxcor >= ((- screen-edge-x) + goal-pos)) and
       (pxcor < ((- screen-edge-x) + goal-pos + goal-size))
     [set pcolor green]
  ]
  set try-line patches with [ (pxcor = ((- screen-edge-x) + kick-line)) and
                              (abs pycor < screen-edge-y) ]
  ask try-line [ set pcolor yellow ]
  set histogram-area
    patches with [ (pxcor < (- screen-edge-x) + kick-line) and
                   (pxcor > (- screen-edge-x)) and
                   ((abs pycor) < screen-edge-y) ]
end

;; turtle procedure that resets all balls to kick line at end of each round
to setup-balls  
  set-default-shape turtles "circle"
  cct number [
    set color orange
    setxy ((- screen-edge-x) + kick-line)
          (1 - screen-edge-y + random (screen-size-y - 2))
    set start-patch patch-here
    set heading 1
    rt random-float 90
  ]
  plot-scores
end

to go
  ask turtles [ move ]
  if not any? turtles
    [ set kicks kicks + number 
      set goals sum values-from try-line [score] 
      setup-balls ]
end

;; turtle procedure that moves all balls
to move
  fd 1
  if (pcolor = red)
    [ die ]       ;; the ball has hit the border wall
  if (pcolor = green) ;; the ball has reached the goal
    [ ;; increment the number of times a goal has been scored from this point on the kick line
      ask start-patch
        [ set score score + 1 ]
      die ]
end

;; do on-screen histogramming
to plot-scores
  ;; set the maximum goals scored from any patch
  set current-max (max values-from try-line [score])
  if current-max = 0
    [ ask histogram-area [ set pcolor black ]
      stop  ; otherwise we'll get division-by-zero errors below
    ]
  ask try-line [
    ifelse score = current-max
      [ set best-dist (screen-edge-y - pycor)
        set (plabel-of patch-at 2 0) (screen-edge-y - pycor) ]
      [ if pcolor != magenta
          [ set (plabel-of patch-at 2 0) no-label ] ]
  ]
  ask histogram-area
  [;; make the histogram bars
   ifelse (((- screen-edge-x) + kick-line) - pxcor) <
          ((score-of patch-at (((- screen-edge-x) + kick-line) - pxcor) 0) *
           kick-line / current-max)
    ;; make the pink histogram bars at the maximal locations
    [ifelse (score-of patch-at (((- screen-edge-x) + kick-line) - pxcor) 0
              = current-max) and
            (((- screen-edge-x) + kick-line) - pxcor) <
            ((score-of patch-at (((- screen-edge-x) + kick-line) - pxcor) 0) *
             kick-line / current-max)
      [set pcolor yellow]
      ;; other locations get blue bars
      [set pcolor blue] ]
    [set pcolor black]
  ]
end

to find-analytic-solution
  ask patches with [pycor < screen-edge-y]
    [ calc-goal-angle ]
  ;; calculate the analytic solution for best kicking point
  let winning-patch max-one-of try-line [goal-angle]
  ask winning-patch
    [ set pcolor magenta
      set (plabel-of patch-at 2 0) (screen-edge-y - pycor) ]
  set analytic value-from winning-patch [screen-edge-y - pycor]
end

to draw-level-curves
  ask patches with [(pxcor > ((- screen-edge-x) + kick-line)) and
                    (pcolor < 10)]
    [ if goal-angle < 90
       [ set pcolor (goal-angle mod 10) * 0.8 ] ]
end

;; calculate angle between patch and goal
to calc-goal-angle
  set left-angle  towardsxy-nowrap ((- screen-edge-x) + goal-pos - 0.5)
                                   (screen-edge-y - 0.5)
  set right-angle towardsxy-nowrap ((- screen-edge-x) + goal-pos + goal-size - 0.5)
                                   (screen-edge-y - 0.5)
  set goal-angle (right-angle - left-angle) mod 360
end


; *** NetLogo Model Copyright Notice ***
;
; This model was created as part of the project: CONNECTED MATHEMATICS:
; MAKING SENSE OF COMPLEX PHENOMENA THROUGH BUILDING OBJECT-BASED PARALLEL
; MODELS (OBPML).  The project gratefully acknowledges the support of the
; National Science Foundation (Applications of Advanced Technologies
; Program) -- grant numbers RED #9552950 and REC #9632612.
;
; Copyright 1998 by Uri Wilensky. All rights reserved.
;
; Permission to use, modify or redistribute this model is hereby granted,
; provided that both of the following requirements are followed:
; a) this copyright notice is included.
; b) this model will not be redistributed for profit without permission
;    from Uri Wilensky.
; Contact Uri Wilensky for appropriate licenses for redistribution for
; profit.
;
; This model was converted to NetLogo as part of the project:
; PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN
; CLASSROOMS.  The project gratefully acknowledges the support of the
; National Science Foundation (REPP program) -- grant number REC #9814682.
; Converted from StarLogoT to NetLogo, 2001.  Updated 2003.
;
; To refer to this model in academic publications, please use:
; Wilensky, U. (1998).  NetLogo Rugby model.
; http://ccl.northwestern.edu/netlogo/models/Rugby.
; Center for Connected Learning and Computer-Based Modeling,
; Northwestern University, Evanston, IL.
;
; In other publications, please use:
; Copyright 1998 by Uri Wilensky.  All rights reserved.  See
; http://ccl.northwestern.edu/netlogo/models/Rugby
; for terms of use.
;
; *** End of NetLogo Model Copyright Notice ***
@#$#@#$#@
GRAPHICS-WINDOW
317
10
693
527
30
40
6.0
0
10
1
1
1
0

CC-WINDOW
5
541
702
636
Command Center

BUTTON
93
45
148
78
setup
setup
NIL
1
T
OBSERVER
T
NIL

SLIDER
18
126
155
159
kick-line
kick-line
2
40
20
1
1
NIL

SLIDER
82
163
219
196
goal-size
goal-size
1
22
11
1
1
NIL

SLIDER
159
126
296
159
goal-pos
goal-pos
1
49
40
1
1
NIL

SLIDER
10
89
300
122
number
number
1
2000
200
1
1
NIL

BUTTON
160
45
215
78
go
go
T
1
T
OBSERVER
NIL
NIL

MONITOR
18
265
178
314
best distance (experimental)
best-dist
2
1

MONITOR
18
315
178
364
best distance (analytic)
analytic
2
1

SWITCH
63
223
239
256
show-level-curves?
show-level-curves?
0
1
-1000

MONITOR
194
265
295
314
NIL
kicks
0
1

MONITOR
194
315
295
364
NIL
goals
0
1

@#$#@#$#@
WHAT IS IT?
-----------
"In rugby, after a try has been scored, the scoring team has the opportunity to gain further points by 'kicking a conversion'. The kick can be taken from anywhere on an imaginary line that is perpendicular to the try line (aka the goal line) and goes through the location on the try line where the try was scored.  Where should the kick be taken from to maximize the chance of a score?"

Instead of trying to solve this problem with calculus or geometry, we'll take a probabilistic approach and use NetLogo turtles and patches. Essentially, we'll have a player stand on each patch along the imaginary kick line (the vertical yellow line), and kick many balls in random directions.  Players receive one 'point' for each conversion they score.  The idea is that since each kick-angle is randomly chosen, those players who score the most points must be standing in the best positions.

Note that this approach is quite different from the analytic approach. In the standard mathematical solution, one is expected to equate the best chance of scoring with the maximal angle from the kick line to the goal. In the probabilistic solution, this is not assumed. It emerges as the result if the playing field is assumed to be uniform. If, however, the model is extended to include wind or grass grain or decaying kick speeds, this solution adjusts gracefully in contrast to the brittle analytic solution.


HOW TO USE IT
-------------
We use four sliders to initialize the model:
- NUMBER is the number of balls that will be kicked in a single round.
- GOAL-SIZE determines the size of the goal.
- GOAL-POS is the x-coordinate of the left goal-post. It is placed GOAL-POS units away from the leftmost screen-edge.
- KICK-LINE determines the x-coordinate of the kick-line. The kick line is KICK-LINE units away from the leftmost screen-edge.

The left goal-post is at ((- screen-edge-x) + GOAL-POS - 0.5, screen-edge-y - 0.5), and the right goal-post is at ((- screen-edge-x) + GOAL-POS + GOAL-SIZE - 0.5, screen-edge-y - 0.5). (0.5 is the horizontal or vertical distance between the center of the patch and its edge; this is needed because the goal posts are effectively in the corners of their patches.)

Note that the values of KICK-LINE, GOAL-SIZE, and GOAL-POS may need to be adjusted to fit your current screen-size.

Press the SETUP button to initialize the model, after you've chosen appropriate values for the sliders. When you're ready to run it, press the GO forever-button. This will commence the kicking; rounds will be repeated and results accumulated until you stop the GO button. Two monitors show the distance of the winning player, and the distance the analytic solution says should be the winner. Two other monitors display how many balls have been kicked and how many resulted in goals.

Watch the left-hand side of the playing field as the model runs. A histogram is being plotted of total points scored from each position along the kick line. The best scores (to date) for each round are highlighted in yellow (and their distance fromm the goal-line is displayed just to their right)-- the others are drawn in blue.The patch that is theoretically calculated to have the best score is shown in magenta and its distance is also displayed alongside it to the right.

In addition to the sequence of rounds described above, if you turn on the SHOW-LEVEL-CURVES? slider and press SETUP, you will see the level-curves associated with the given set of slider settings. This allows you to visualize the field of solutions for the analytic case of a uniform field. Patches are colored according to how large the goal looks from that position. Along each connected curve of the same color, the goal appears to be the same size. From straight ahead, the goal appears maximally wide.  From a shallow angle, the goal looks smaller at the same distance along the try line.


THINGS TO NOTICE
----------------
Given a particular goal-line and kick-line, what are the best positions to kick from? Change the position and size of the goal. How do the 'best positions' change in relation to the goal's position? Then change the position of the kick line, and redo the trial. How does the performance compare when the kick line is farther away? Or closer?

Examine the histogram at the left edge of the screen. What do you notice about it? What shape does it have? What does it tell you about the best locations to kick from?

How does the experimental solution compare to the analytic solution? Given enough time, will the experimental solution converge to the analytic solution? If so, how long does it take?


THINGS TO TRY
-------------
Can you generalize the results given by the model? What are the relationships between the kick-line, the goal, and the best patch to kick from?

Re-read the problem description given in the first paragraph of "What Is It?" Do you think that this model adequately answers the question asked? Why or why not?  What alternative solutions could you provide?

It's important to understand that this NetLogo project isn't exactly the standard method for solving problems of this sort. (A more 'classical' approach would be to use techniques from geometry or calculus.) However, there are many advantages offered by the NetLogo method. Consider the following other variables that might affect the solution to the problem: wind speed, grass height, or the size/weight/skill of the player kicking. (Such a problem that considers 'real-world' parameters such as these may quickly become intractable under classical mathematics.) Try and think of how you'd solve the 'rugby' problem in its idealized version without using NetLogo. Then try and solve it, taking into account one or more of these extra features.

In general, what advantages does the NetLogo solution have over the other methods? What disadvantages does it have?


EXTENDING THE MODEL
-------------------
Implement some of the features descibed above under "Things To Try"- e.g. wind speed, or the size of the player to kick the conversion from a particular position.

There are a variety of plots you could have 'rugby' draw in a plot. Implement plotting procedures for some or all of the following: the number of successful kicks compared to the overall kicks, the plot of both types of kicks over time, or the difference in histograms depending on the locations of the kick-line (i.e. the value of KICK-LINE) and the goal-posts (GOAL-POS and GOAL-SIZE).

You may have noticed that even if you run the model for a long time, such as overnight, the resulting histogram is not perfectly smooth, and that the "bumps" always appear in the same places.  This is because the algorithm used to determine whether a kick succeeds or not isn't perfect.  If you look at the MOVE procedure, you'll see that a ball moves using the command "fd 1".  Since the ball isn't moving continuously, but is jumping in steps of one patch length at a time, this means that if the ball is near to the edge of the goal, it may jump over the corner of the goal, causing a successful kick to be counted as a failure, or vice versa.  You can verify that the bumpiness in the histogram comes from the step size by changing "fd 1" to "fd 0.1" and running the model overnight again; now the graph is much less bumpy.  You can make the graph as smooth as you want by using even smaller step sizes.  But the smaller the step size, the slower the model runs.  Can you figure out a different method for getting a smooth result curve, that doesn't slow the model down drastically?  Hint: you'll need to study the code closely and figure out exactly what is happening at the edges of the goal.  It helps to draw pictures.


NETLOGO FEATURES
----------------
We draw our histogram straight to the graphics window, unlike most other NetLogo models where the histogram is drawn in a separate plot.


CREDITS AND REFERENCES
----------------------
The problem is taken from a British mathematics textbook.

For a fuller discussion, see Wilensky, U. (1996). Modeling Rugby: Kick First, Generalize Later? International Journal of Computers for Mathematical Learning. Vol. 1,  No. 1. p. 124 - 131.   http://ccl.northwestern.edu/cm/papers/rugby/

To refer to this model in academic publications, please use: Wilensky, U. (1998).  NetLogo Rugby model. http://ccl.northwestern.edu/netlogo/models/Rugby. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

In other publications, please use: Copyright 1998 by Uri Wilensky.  All rights reserved.  See http://ccl.northwestern.edu/netlogo/models/Rugby for terms of use.
@#$#@#$#@
default
true
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7566196 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7566196 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7566196 true true 150 285 285 225 285 75 150 135
Polygon -7566196 true true 150 135 15 75 150 15 285 75
Polygon -7566196 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7566196 true true 96 182 108
Circle -7566196 true true 110 127 80
Circle -7566196 true true 110 75 80
Line -7566196 true 150 100 80 30
Line -7566196 true 150 100 220 30

butterfly
true
0
Polygon -7566196 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7566196 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7566196 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7566196 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7566196 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7566196 true true 47 195 58
Circle -7566196 true true 195 195 58

circle
false
0
Circle -7566196 true true 30 30 240

circle 2
false
0
Circle -7566196 true true 16 16 270
Circle -16777216 true false 46 46 210

cow
false
0
Polygon -7566196 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7566196 true true 73 210 86 251 62 249 48 208
Polygon -7566196 true true 25 114 16 195 9 204 23 213 25 200 39 123

face happy
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7566196 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7566196 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7566196 true true 60 15 75 300
Polygon -7566196 true true 90 150 270 90 90 30
Line -7566196 true 75 135 90 135
Line -7566196 true 75 45 90 45

flower
false
0
Polygon -11352576 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7566196 true true 85 132 38
Circle -7566196 true true 130 147 38
Circle -7566196 true true 192 85 38
Circle -7566196 true true 85 40 38
Circle -7566196 true true 177 40 38
Circle -7566196 true true 177 132 38
Circle -7566196 true true 70 85 38
Circle -7566196 true true 130 25 38
Circle -7566196 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -11352576 true false 189 233 219 188 249 173 279 188 234 218
Polygon -11352576 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7566196 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7566196 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7566196 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7566196 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7566196 true 150 0 150 300

pentagon
false
0
Polygon -7566196 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7566196 true true 110 5 80
Polygon -7566196 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7566196 true true 127 79 172 94
Polygon -7566196 true true 195 90 240 150 225 180 165 105
Polygon -7566196 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7566196 true true 135 90 165 300
Polygon -7566196 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7566196 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7566196 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7566196 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7566196 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7566196 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7566196 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7566196 true true 30 30 270 270

square 2
false
0
Rectangle -7566196 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7566196 true true 60 270 150 0 240 270 15 105 285 105
Polygon -7566196 true true 75 120 105 210 195 210 225 120 150 75

target
false
0
Circle -7566196 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7566196 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7566196 true true 120 120 60

tree
false
0
Circle -7566196 true true 118 3 94
Rectangle -6524078 true false 120 195 180 300
Circle -7566196 true true 65 21 108
Circle -7566196 true true 116 41 127
Circle -7566196 true true 45 90 120
Circle -7566196 true true 104 74 152

triangle
false
0
Polygon -7566196 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7566196 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7566196 true true 4 45 195 187
Polygon -7566196 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7566196 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7566196 false true 24 174 42
Circle -7566196 false true 144 174 42
Circle -7566196 false true 234 174 42

turtle
true
0
Polygon -11352576 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -11352576 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -11352576 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -11352576 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -11352576 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7566196 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7566196 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7566196 true 150 285 150 15
Line -7566196 true 15 150 285 150
Circle -7566196 true true 120 120 60
Line -7566196 true 216 40 79 269
Line -7566196 true 40 84 269 221
Line -7566196 true 40 216 269 79
Line -7566196 true 84 40 221 269

x
false
0
Polygon -7566196 true true 270 75 225 30 30 225 75 270
Polygon -7566196 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 2.1beta4
@#$#@#$#@
setup
repeat 16000 [ go ]
@#$#@#$#@
@#$#@#$#@
